<?php
include 'layouts/session.php';
include 'layouts/dbconn.php';

header('Content-Type: application/json');

if ($_SERVER["REQUEST_METHOD"] == "POST") {
    $conn->begin_transaction();
    
    try {
        // Validate required fields
        $requiredFields = ['client_name', 'reference_number', 'amount', 'payment', 'currency', 'staff_id'];
        
        foreach ($requiredFields as $field) {
            if (!isset($_POST[$field]) || empty($_POST[$field])) {
                throw new Exception("Missing required field: " . $field);
            }
        }
        
        // Validate staff_id
        $staffId = filter_var($_POST['staff_id'], FILTER_VALIDATE_INT);
        if (!$staffId) {
            throw new Exception("Invalid staff selection");
        }
        
        // Verify staff exists and is active
        $staffCheck = $conn->prepare("SELECT staff_id FROM staff WHERE staff_id = ? AND status = 'active'");
        $staffCheck->bind_param("i", $staffId);
        $staffCheck->execute();
        
        if ($staffCheck->get_result()->num_rows === 0) {
            throw new Exception("Invalid or inactive staff member selected");
        }
        $staffCheck->close();
        
        // Validate amount and payment
        $amount = floatval($_POST['amount']);
        $payment = floatval($_POST['payment']);
        
        if ($amount <= 0) {
            throw new Exception("Amount must be greater than zero");
        }
        
        if ($payment < 0) {
            throw new Exception("Payment cannot be negative");
        }
        
        if ($payment > $amount) {
            throw new Exception("Payment cannot be greater than amount");
        }
        
        // Insert visa application
        $stmt = $conn->prepare("INSERT INTO visa_applications (
            client_name, staff_id, reference_number, care_of, amount, payment,
            currency, notes, created_by, status
        ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, 'Pending')");
        
        $createdBy = $_SESSION['user_id'] ?? null;
        $careOf = !empty($_POST['care_of']) ? $_POST['care_of'] : null;
        
        $stmt->bind_param("sissddssi",
            $_POST['client_name'],
            $staffId,
            $_POST['reference_number'],
            $careOf,
            $amount,
            $payment,
            $_POST['currency'],
            $_POST['notes'],
            $createdBy
        );
        
        if (!$stmt->execute()) {
            throw new Exception("Error saving visa application: " . $stmt->error);
        }
        
        $applicationId = $conn->insert_id;
        $stmt->close();
        
        // Get staff name for audit log
        $staffNameQuery = $conn->prepare("SELECT full_name FROM staff WHERE staff_id = ?");
        $staffNameQuery->bind_param("i", $staffId);
        $staffNameQuery->execute();
        $staffResult = $staffNameQuery->get_result();
        $staffName = $staffResult->fetch_assoc()['full_name'];
        
        // Add audit log
        $logStmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, log_details, ip_address, user_agent) 
                                  VALUES (?, 'CREATE_VISA', ?, ?, ?)");
        $logDetails = "Created visa application for " . $_POST['client_name'] . " (Staff: " . $staffName . ")";
        $logStmt->bind_param(
            "isss", 
            $createdBy,
            $logDetails,
            $_SERVER['REMOTE_ADDR'],
            $_SERVER['HTTP_USER_AGENT']
        );
        $logStmt->execute();
        
        $conn->commit();
        
        echo json_encode([
            'status' => 'success',
            'message' => 'Visa application saved successfully!',
            'application_id' => $applicationId
        ]);
        
    } catch (Exception $e) {
        $conn->rollback();
        echo json_encode([
            'status' => 'error',
            'message' => $e->getMessage()
        ]);
    }
}

$conn->close();
?>