<?php
session_start();
include 'layouts/dbconn.php';

if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Validate required fields
    $required_fields = [
        'extension_date', 'client_name', 'passport_number', 
        'country_of_issue', 'service_type', 'validity_period',
        'expiry_date', 'amount_received', 'service_payment', 
        'payment_status', 'currency', 'staff_id'
    ];
    
    foreach ($required_fields as $field) {
        if (empty($_POST[$field])) {
            throw new Exception("$field is required");
        }
    }
    
    // Validate dates
    $extension_date = new DateTime($_POST['extension_date']);
    $expiry_date = new DateTime($_POST['expiry_date']);
    
    if ($expiry_date <= $extension_date) {
        throw new Exception("Expiry date must be after extension date");
    }
    
    // Validate amounts
    $amount_received = floatval($_POST['amount_received']);
    $service_payment = floatval($_POST['service_payment']);
    
    if ($amount_received < 0 || $service_payment < 0) {
        throw new Exception("Amounts cannot be negative");
    }
    
    if ($service_payment > $amount_received) {
        throw new Exception("Service payment cannot be greater than amount received");
    }
    
    // Validate staff_id
    $staff_id = filter_var($_POST['staff_id'], FILTER_VALIDATE_INT);
    if (!$staff_id) {
        throw new Exception("Invalid staff selection");
    }
    
    // Verify staff exists and is active
    $staff_check = $conn->prepare("SELECT staff_id FROM staff WHERE staff_id = ? AND status = 'active'");
    $staff_check->bind_param("i", $staff_id);
    $staff_check->execute();
    
    if ($staff_check->get_result()->num_rows === 0) {
        throw new Exception("Invalid or inactive staff member selected");
    }
    $staff_check->close();
    
    // Prepare the SQL statement
    $sql = "INSERT INTO passport_extensions (
                extension_date, client_name, passport_number, 
                country_of_issue, service_type, validity_period,
                expiry_date, amount_received, service_payment,
                payment_status, notes, created_by, currency, staff_id
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";
    
    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        throw new Exception("Failed to prepare statement: " . $conn->error);
    }
    
    // Bind parameters
    $stmt->bind_param(
        "sssssisddssisi",
        $_POST['extension_date'],
        $_POST['client_name'],
        $_POST['passport_number'],
        $_POST['country_of_issue'],
        $_POST['service_type'],
        $_POST['validity_period'],
        $_POST['expiry_date'],
        $amount_received,
        $service_payment,
        $_POST['payment_status'],
        $_POST['notes'],
        $_SESSION['user_id'],
        $_POST['currency'],
        $staff_id
    );
    
    // Execute the statement
    if (!$stmt->execute()) {
        throw new Exception("Failed to save extension: " . $stmt->error);
    }
    
    $extension_id = $conn->insert_id;
    
    // Log the action
    $log_sql = "INSERT INTO audit_logs (user_id, action, log_details, ip_address, user_agent) 
                VALUES (?, 'CREATE_EXTENSION', ?, ?, ?)";
    $log_stmt = $conn->prepare($log_sql);
    
    // Get staff name for log details
    $staff_name_query = $conn->prepare("SELECT full_name FROM staff WHERE staff_id = ?");
    $staff_name_query->bind_param("i", $staff_id);
    $staff_name_query->execute();
    $staff_result = $staff_name_query->get_result();
    $staff_name = $staff_result->fetch_assoc()['full_name'];
    
    $log_details = "Created passport extension for " . $_POST['client_name'] . " (Staff: " . $staff_name . ")";
    $log_stmt->bind_param(
        "isss", 
        $_SESSION['user_id'],
        $log_details,
        $_SERVER['REMOTE_ADDR'],
        $_SERVER['HTTP_USER_AGENT']
    );
    $log_stmt->execute();
    
    echo json_encode([
        'success' => true,
        'message' => 'Passport extension saved successfully',
        'extension_id' => $extension_id
    ]);
    
} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

$conn->close();
?>