<?php
require_once __DIR__ . '/vendor/phpoffice/vendor/autoload.php';
include 'layouts/session.php';
include 'layouts/dbconn.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;
use PhpOffice\PhpSpreadsheet\Style\Alignment;

// Get filter parameters
$staff_id = isset($_GET['staff_id']) ? $_GET['staff_id'] : '';
$service_type = isset($_GET['service_type']) ? $_GET['service_type'] : 'all';
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Get staff name if selected
$selected_staff_name = '';
if (!empty($staff_id)) {
    $staff_name_sql = "SELECT full_name FROM staff WHERE staff_id = ?";
    $stmt = $conn->prepare($staff_name_sql);
    $stmt->bind_param("i", $staff_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $selected_staff_name = $row['full_name'];
    }
}

$spreadsheet = new Spreadsheet();

// Remove the default sheet
$spreadsheet->removeSheetByIndex(0);

// If it's all services, create a summary sheet first
if ($service_type == 'all') {
    createSummarySheet($spreadsheet, $conn, $staff_id, $start_date, $end_date, $selected_staff_name);
}

// Create sheets based on service type
if ($service_type == 'all' || $service_type == 'tickets') {
    createTicketSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date);
}
if ($service_type == 'all' || $service_type == 'visas') {
    createVisaSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date);
}
if ($service_type == 'all' || $service_type == 'extensions') {
    createExtensionSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date);
}
if ($service_type == 'all' || $service_type == 'go_home') {
    createGoHomeSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date);
}

// Set active sheet to the first sheet
$spreadsheet->setActiveSheetIndex(0);

// Function to create tickets sheet
function createTicketSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date) {
    $sheet = $spreadsheet->createSheet();
    $sheet->setTitle('Ticket Sales');
    
    // Set headers
    $sheet->setCellValue('A1', 'Staff Services Report - Ticket Sales');
    $sheet->mergeCells('A1:I1');
    $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
    $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
    
    $sheet->setCellValue('A2', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
    $sheet->mergeCells('A2:I2');
    
    // Column headers
    $headers = ['Date', 'Staff', 'Customer', 'Ticket Reference', 'Airline', 'Route', 'Ticket Cost', 'Agency Fee', 'Total'];
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '4', $header);
        $col++;
    }
    
    // Style headers
    $sheet->getStyle('A4:I4')->getFont()->setBold(true);
    $sheet->getStyle('A4:I4')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');
    
    // Get ticket data
    $sql = "SELECT td.*, s.full_name as staff_name
            FROM ticket_details td
            LEFT JOIN staff s ON td.staff_id = s.staff_id
            WHERE DATE(td.flight_date) BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND td.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY td.flight_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $row = 5;
    $total_ticket_cost = 0;
    $total_ticket_commission = 0;
    
    while ($ticket = $result->fetch_assoc()) {
        $sheet->setCellValue('A' . $row, date('d M Y', strtotime($ticket['flight_date'])));
        $sheet->setCellValue('B' . $row, $ticket['staff_name'] ?? 'N/A');
        $sheet->setCellValue('C' . $row, $ticket['customer_name']);
        $sheet->setCellValue('D' . $row, $ticket['ticket_ref']);
        $sheet->setCellValue('E' . $row, $ticket['airline']);
        $sheet->setCellValue('F' . $row, $ticket['flight_route']);
        $sheet->setCellValue('G' . $row, $ticket['ticket_cost']);
        $sheet->setCellValue('H' . $row, $ticket['agency_fee']);
        $sheet->setCellValue('I' . $row, $ticket['total_amount']);
        
        $total_ticket_cost += $ticket['ticket_cost'];
        $total_ticket_commission += $ticket['agency_fee'];
        
        $row++;
    }
    
    // Add totals
    $sheet->setCellValue('A' . $row, 'Total');
    $sheet->mergeCells('A' . $row . ':F' . $row);
    $sheet->setCellValue('G' . $row, $total_ticket_cost);
    $sheet->setCellValue('H' . $row, $total_ticket_commission);
    $sheet->setCellValue('I' . $row, $total_ticket_cost + $total_ticket_commission);
    $sheet->getStyle('A' . $row . ':I' . $row)->getFont()->setBold(true);
    
    // Format numbers
    $sheet->getStyle('G5:I' . $row)->getNumberFormat()->setFormatCode('#,##0.00');
    
    // Auto-size columns
    foreach(range('A','I') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
    
    // Add borders
    $sheet->getStyle('A4:I' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
}

// Function to create visa sheet
function createVisaSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date) {
    $sheet = $spreadsheet->createSheet();
    $sheet->setTitle('Visa Applications');
    
    // Set headers
    $sheet->setCellValue('A1', 'Staff Services Report - Visa Applications');
    $sheet->mergeCells('A1:H1');
    $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
    $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
    
    $sheet->setCellValue('A2', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
    $sheet->mergeCells('A2:H2');
    
    // Column headers
    $headers = ['Date', 'Staff', 'Client Name', 'Reference Number', 'Amount', 'Payment', 'Commission', 'Status'];
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '4', $header);
        $col++;
    }
    
    // Style headers
    $sheet->getStyle('A4:H4')->getFont()->setBold(true);
    $sheet->getStyle('A4:H4')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');
    
    // Get visa data
    $sql = "SELECT va.*, s.full_name as staff_name
            FROM visa_applications va
            LEFT JOIN staff s ON va.staff_id = s.staff_id
            WHERE va.created_at BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND va.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY va.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $row = 5;
    $total_visa_amount = 0;
    $total_visa_commission = 0;
    
    while ($visa = $result->fetch_assoc()) {
        $sheet->setCellValue('A' . $row, date('d M Y', strtotime($visa['created_at'])));
        $sheet->setCellValue('B' . $row, $visa['staff_name'] ?? 'N/A');
        $sheet->setCellValue('C' . $row, $visa['client_name']);
        $sheet->setCellValue('D' . $row, $visa['reference_number']);
        $sheet->setCellValue('E' . $row, $visa['amount']);
        $sheet->setCellValue('F' . $row, $visa['payment']);
        $sheet->setCellValue('G' . $row, $visa['commission']);
        $sheet->setCellValue('H' . $row, ucfirst($visa['status']));
        
        $total_visa_amount += $visa['amount'];
        $total_visa_commission += $visa['commission'];
        
        $row++;
    }
    
    // Add totals
    $sheet->setCellValue('A' . $row, 'Total');
    $sheet->mergeCells('A' . $row . ':D' . $row);
    $sheet->setCellValue('E' . $row, $total_visa_amount);
    $sheet->setCellValue('F' . $row, '-');
    $sheet->setCellValue('G' . $row, $total_visa_commission);
    $sheet->getStyle('A' . $row . ':H' . $row)->getFont()->setBold(true);
    
    // Format numbers
    $sheet->getStyle('E5:G' . $row)->getNumberFormat()->setFormatCode('#,##0.00');
    
    // Auto-size columns
    foreach(range('A','H') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
    
    // Add borders
    $sheet->getStyle('A4:H' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
}

// Function to create extension sheet
function createExtensionSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date) {
    $sheet = $spreadsheet->createSheet();
    $sheet->setTitle('Passport Extensions');
    
    // Set headers
    $sheet->setCellValue('A1', 'Staff Services Report - Passport Extensions');
    $sheet->mergeCells('A1:I1');
    $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
    $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
    
    $sheet->setCellValue('A2', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
    $sheet->mergeCells('A2:I2');
    
    // Column headers
    $headers = ['Date', 'Staff', 'Client Name', 'Passport Number', 'Type', 'Amount Received', 'Service Payment', 'Agency Fee', 'Status'];
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '4', $header);
        $col++;
    }
    
    // Style headers
    $sheet->getStyle('A4:I4')->getFont()->setBold(true);
    $sheet->getStyle('A4:I4')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');
    
    // Get extension data
    $sql = "SELECT pe.*, s.full_name as staff_name
            FROM passport_extensions pe
            LEFT JOIN staff s ON pe.staff_id = s.staff_id
            WHERE pe.extension_date BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND pe.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY pe.extension_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $row = 5;
    $total_ext_amount = 0;
    $total_ext_agency_fee = 0;
    
    while ($ext = $result->fetch_assoc()) {
        $sheet->setCellValue('A' . $row, date('d M Y', strtotime($ext['extension_date'])));
        $sheet->setCellValue('B' . $row, $ext['staff_name'] ?? 'N/A');
        $sheet->setCellValue('C' . $row, $ext['client_name']);
        $sheet->setCellValue('D' . $row, $ext['passport_number']);
        $sheet->setCellValue('E' . $row, ucfirst($ext['service_type']));
        $sheet->setCellValue('F' . $row, $ext['amount_received']);
        $sheet->setCellValue('G' . $row, $ext['service_payment']);
        $sheet->setCellValue('H' . $row, $ext['agency_fee']);
        $sheet->setCellValue('I' . $row, ucfirst($ext['payment_status']));
        
        $total_ext_amount += $ext['amount_received'];
        $total_ext_agency_fee += $ext['agency_fee'];
        
        $row++;
    }
    
    // Add totals
    $sheet->setCellValue('A' . $row, 'Total');
    $sheet->mergeCells('A' . $row . ':E' . $row);
    $sheet->setCellValue('F' . $row, $total_ext_amount);
    $sheet->setCellValue('G' . $row, '-');
    $sheet->setCellValue('H' . $row, $total_ext_agency_fee);
    $sheet->getStyle('A' . $row . ':I' . $row)->getFont()->setBold(true);
    
    // Format numbers
    $sheet->getStyle('F5:H' . $row)->getNumberFormat()->setFormatCode('#,##0.00');
    
    // Auto-size columns
    foreach(range('A','I') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
    
    // Add borders
    $sheet->getStyle('A4:I' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
}

// Function to create go home sheet
function createGoHomeSheet($spreadsheet, $conn, $staff_id, $start_date, $end_date) {
    $sheet = $spreadsheet->createSheet();
    $sheet->setTitle('Go Home Services');
    
    // Set headers
    $sheet->setCellValue('A1', 'Staff Services Report - Go Home Services');
    $sheet->mergeCells('A1:I1');
    $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
    $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
    
    $sheet->setCellValue('A2', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
    $sheet->mergeCells('A2:I2');
    
    // Column headers
    $headers = ['Date', 'Staff', 'Client Name', 'Mother Name', 'DOB', 'Reference', 'Price', 'Office Fee', 'Status'];
    $col = 'A';
    foreach ($headers as $header) {
        $sheet->setCellValue($col . '4', $header);
        $col++;
    }
    
    // Style headers
    $sheet->getStyle('A4:I4')->getFont()->setBold(true);
    $sheet->getStyle('A4:I4')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');
    
    // Get go home data
    $sql = "SELECT ghs.*, s.full_name as staff_name
            FROM go_home_service ghs
            LEFT JOIN staff s ON ghs.staff_id = s.staff_id
            WHERE ghs.service_date BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND ghs.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY ghs.service_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    
    $row = 5;
    $total_go_home_price = 0;
    $total_go_home_fee = 0;
    
    while ($ghs = $result->fetch_assoc()) {
        $sheet->setCellValue('A' . $row, date('d M Y', strtotime($ghs['service_date'])));
        $sheet->setCellValue('B' . $row, $ghs['staff_name'] ?? 'N/A');
        $sheet->setCellValue('C' . $row, $ghs['client_name']);
        $sheet->setCellValue('D' . $row, $ghs['mother_name']);
        $sheet->setCellValue('E' . $row, date('d M Y', strtotime($ghs['date_of_birth'])));
        $sheet->setCellValue('F' . $row, $ghs['reference_number']);
        $sheet->setCellValue('G' . $row, $ghs['price']);
        $sheet->setCellValue('H' . $row, $ghs['office_fee']);
        $sheet->setCellValue('I' . $row, ucfirst($ghs['status']));
        
        $total_go_home_price += $ghs['price'];
        $total_go_home_fee += $ghs['office_fee'];
        
        $row++;
    }
    
    // Add totals
    $sheet->setCellValue('A' . $row, 'Total');
    $sheet->mergeCells('A' . $row . ':F' . $row);
    $sheet->setCellValue('G' . $row, $total_go_home_price);
    $sheet->setCellValue('H' . $row, $total_go_home_fee);
    $sheet->getStyle('A' . $row . ':I' . $row)->getFont()->setBold(true);
    
    // Format numbers
    $sheet->getStyle('G5:H' . $row)->getNumberFormat()->setFormatCode('#,##0.00');
    
    // Auto-size columns
    foreach(range('A','I') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
    
    // Add borders
    $sheet->getStyle('A4:I' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
}

// Function to create summary sheet
function createSummarySheet($spreadsheet, $conn, $staff_id, $start_date, $end_date, $selected_staff_name) {
    $sheet = $spreadsheet->createSheet();
    $sheet->setTitle('Summary');
    
    // Set headers
    $sheet->setCellValue('A1', 'Staff Services Report - Summary');
    $sheet->mergeCells('A1:D1');
    $sheet->getStyle('A1')->getFont()->setBold(true)->setSize(14);
    $sheet->getStyle('A1')->getAlignment()->setHorizontal(Alignment::HORIZONTAL_CENTER);
    
    if (!empty($selected_staff_name)) {
        $sheet->setCellValue('A2', 'Staff: ' . $selected_staff_name);
        $sheet->mergeCells('A2:D2');
    }
    
    $sheet->setCellValue('A3', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
    $sheet->mergeCells('A3:D3');
    
    // Summary headers
    $sheet->setCellValue('A5', 'Service Type');
    $sheet->setCellValue('B5', 'Total Revenue');
    $sheet->setCellValue('C5', 'Total Commission');
    $sheet->setCellValue('D5', 'Total Transactions');
    
    $sheet->getStyle('A5:D5')->getFont()->setBold(true);
    $sheet->getStyle('A5:D5')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');
    
    $row = 6;
    
    // Get summary data
    $totals = getSummaryTotals($conn, $staff_id, $start_date, $end_date);
    
    // Tickets
    $sheet->setCellValue('A' . $row, 'Ticket Sales');
    $sheet->setCellValue('B' . $row, $totals['tickets']['revenue']);
    $sheet->setCellValue('C' . $row, $totals['tickets']['commission']);
    $sheet->setCellValue('D' . $row, $totals['tickets']['count']);
    $row++;
    
    // Visas
    $sheet->setCellValue('A' . $row, 'Visa Applications');
    $sheet->setCellValue('B' . $row, $totals['visas']['revenue']);
    $sheet->setCellValue('C' . $row, $totals['visas']['commission']);
    $sheet->setCellValue('D' . $row, $totals['visas']['count']);
    $row++;
    
    // Extensions
    $sheet->setCellValue('A' . $row, 'Passport Extensions');
    $sheet->setCellValue('B' . $row, $totals['extensions']['revenue']);
    $sheet->setCellValue('C' . $row, $totals['extensions']['commission']);
    $sheet->setCellValue('D' . $row, $totals['extensions']['count']);
    $row++;
    
    // Go Home
    $sheet->setCellValue('A' . $row, 'Go Home Services');
    $sheet->setCellValue('B' . $row, $totals['go_home']['revenue']);
    $sheet->setCellValue('C' . $row, $totals['go_home']['commission']);
    $sheet->setCellValue('D' . $row, $totals['go_home']['count']);
    $row++;
    
    // Grand Total
    $sheet->setCellValue('A' . $row, 'Grand Total');
    $sheet->setCellValue('B' . $row, $totals['total']['revenue']);
    $sheet->setCellValue('C' . $row, $totals['total']['commission']);
    $sheet->setCellValue('D' . $row, $totals['total']['count']);
    $sheet->getStyle('A' . $row . ':D' . $row)->getFont()->setBold(true);
    
    // Format numbers
    $sheet->getStyle('B6:C' . $row)->getNumberFormat()->setFormatCode('#,##0.00');
    $sheet->getStyle('D6:D' . $row)->getNumberFormat()->setFormatCode('#,##0');
    
    // Auto-size columns
    foreach(range('A','D') as $col) {
        $sheet->getColumnDimension($col)->setAutoSize(true);
    }
    
    // Add borders
    $sheet->getStyle('A5:D' . $row)->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);
}

// Function to get summary totals
function getSummaryTotals($conn, $staff_id, $start_date, $end_date) {
    $totals = [
        'tickets' => ['revenue' => 0, 'commission' => 0, 'count' => 0],
        'visas' => ['revenue' => 0, 'commission' => 0, 'count' => 0],
        'extensions' => ['revenue' => 0, 'commission' => 0, 'count' => 0],
        'go_home' => ['revenue' => 0, 'commission' => 0, 'count' => 0],
        'total' => ['revenue' => 0, 'commission' => 0, 'count' => 0]
    ];
    
    // Get ticket totals
    $sql = "SELECT SUM(ticket_cost) as revenue, SUM(agency_fee) as commission, COUNT(*) as count
            FROM ticket_details td
            WHERE DATE(td.flight_date) BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($staff_id)) {
        $sql .= " AND td.staff_id = ?";
        $params[] = $staff_id;
        $types .= "i";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    $totals['tickets']['revenue'] = $result['revenue'] ?? 0;
    $totals['tickets']['commission'] = $result['commission'] ?? 0;
    $totals['tickets']['count'] = $result['count'] ?? 0;
    
    // Get visa totals
    $sql = "SELECT SUM(amount) as revenue, SUM(commission) as commission, COUNT(*) as count
            FROM visa_applications va
            WHERE va.created_at BETWEEN ? AND ?";
    
    if (!empty($staff_id)) {
        $sql .= " AND va.staff_id = ?";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    $totals['visas']['revenue'] = $result['revenue'] ?? 0;
    $totals['visas']['commission'] = $result['commission'] ?? 0;
    $totals['visas']['count'] = $result['count'] ?? 0;
    
    // Get extension totals
    $sql = "SELECT SUM(amount_received) as revenue, SUM(agency_fee) as commission, COUNT(*) as count
            FROM passport_extensions pe
            WHERE pe.extension_date BETWEEN ? AND ?";
    
    if (!empty($staff_id)) {
        $sql .= " AND pe.staff_id = ?";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    $totals['extensions']['revenue'] = $result['revenue'] ?? 0;
    $totals['extensions']['commission'] = $result['commission'] ?? 0;
    $totals['extensions']['count'] = $result['count'] ?? 0;
    
    // Get go home totals
    $sql = "SELECT SUM(price) as revenue, SUM(office_fee) as commission, COUNT(*) as count
            FROM go_home_service ghs
            WHERE ghs.service_date BETWEEN ? AND ?";
    
    if (!empty($staff_id)) {
        $sql .= " AND ghs.staff_id = ?";
    }
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    
    $totals['go_home']['revenue'] = $result['revenue'] ?? 0;
    $totals['go_home']['commission'] = $result['commission'] ?? 0;
    $totals['go_home']['count'] = $result['count'] ?? 0;
    
    // Calculate grand totals
    foreach (['revenue', 'commission', 'count'] as $key) {
        $totals['total'][$key] = 
            $totals['tickets'][$key] +
            $totals['visas'][$key] +
            $totals['extensions'][$key] +
            $totals['go_home'][$key];
    }
    
    return $totals;
}

// Generate filename
$filename = 'staff_services_report_' . date('Y-m-d');
if (!empty($selected_staff_name)) {
    $filename .= '_' . str_replace(' ', '_', $selected_staff_name);
}
$filename .= '.xlsx';

// Create writer and save
$writer = new Xlsx($spreadsheet);

// Set headers for download
header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="' . $filename . '"');
header('Cache-Control: max-age=0');

$writer->save('php://output');
exit;
?>