<?php
require_once __DIR__ . '/vendor/phpoffice/vendor/autoload.php';
include 'layouts/session.php';
include 'layouts/dbconn.php';

use PhpOffice\PhpSpreadsheet\Spreadsheet;
use PhpOffice\PhpSpreadsheet\Writer\Xlsx;
use PhpOffice\PhpSpreadsheet\Style\Border;
use PhpOffice\PhpSpreadsheet\Style\Fill;

$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');
$currency_filter = isset($_GET['currency']) ? $_GET['currency'] : 'all';

$spreadsheet = new Spreadsheet();
$sheet = $spreadsheet->getActiveSheet();

$sheet->setCellValue('A1', 'Passport Extensions Report');
$sheet->mergeCells('A1:K1');
$sheet->getStyle('A1:K1')->getFont()->setBold(true)->setSize(14);
$sheet->getStyle('A1:K1')->getAlignment()->setHorizontal(\PhpOffice\PhpSpreadsheet\Style\Alignment::HORIZONTAL_CENTER);

$sheet->setCellValue('A2', 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date)));
$sheet->mergeCells('A2:K2');

$sheet->setCellValue('A4', 'Extension ID');
$sheet->setCellValue('B4', 'Date');
$sheet->setCellValue('C4', 'Client Name');
$sheet->setCellValue('D4', 'Reference No');
$sheet->setCellValue('E4', 'Service Type');
$sheet->setCellValue('F4', 'Duration');
$sheet->setCellValue('G4', 'Expiry');
$sheet->setCellValue('H4', 'Amount');
$sheet->setCellValue('I4', 'Payment');
$sheet->setCellValue('J4', 'Agency Fee');
$sheet->setCellValue('K4', 'Status');

$sheet->getStyle('A4:K4')->getFont()->setBold(true);
$sheet->getStyle('A4:K4')->getFill()->setFillType(Fill::FILL_SOLID)->getStartColor()->setRGB('E6E6E6');

$sql = "SELECT pe.*, u_created.full_name as created_by_name
        FROM passport_extensions pe
        LEFT JOIN users u_created ON pe.created_by = u_created.user_id
        WHERE pe.extension_date BETWEEN ? AND ?";

if ($currency_filter !== 'all') {
    $sql .= " AND pe.currency = ?";
}

$stmt = $conn->prepare($sql);
if ($currency_filter !== 'all') {
    $stmt->bind_param("sss", $start_date, $end_date, $currency_filter);
} else {
    $stmt->bind_param("ss", $start_date, $end_date);
}

$stmt->execute();
$result = $stmt->get_result();

$row = 5;
$totals = [];

while ($extension = $result->fetch_assoc()) {
    $sheet->setCellValue('A' . $row, $extension['extension_id']);
    $sheet->setCellValue('B' . $row, date('d M Y', strtotime($extension['extension_date'])));
    $sheet->setCellValue('C' . $row, $extension['client_name']);
    $sheet->setCellValue('D' . $row, $extension['reference_number']);
    $sheet->setCellValue('E' . $row, ucfirst($extension['service_type']));
    $sheet->setCellValue('F' . $row, $extension['validity_period'] . ' months');
    $sheet->setCellValue('G' . $row, date('d M Y', strtotime($extension['expiry_date'])));
    $sheet->setCellValue('H' . $row, $extension['currency'] . ' ' . number_format($extension['amount'], 2));
    $sheet->setCellValue('I' . $row, $extension['currency'] . ' ' . number_format($extension['payment'], 2));
    $sheet->setCellValue('J' . $row, $extension['currency'] . ' ' . number_format($extension['amount'] - $extension['payment'], 2));
    $sheet->setCellValue('K' . $row, ucfirst($extension['status']));

    if (!isset($totals[$extension['currency']])) {
        $totals[$extension['currency']] = [
            'amount' => 0,
            'payment' => 0,
            'agency_fee' => 0
        ];
    }
    $totals[$extension['currency']]['amount'] += $extension['amount'];
    $totals[$extension['currency']]['payment'] += $extension['payment'];
    $totals[$extension['currency']]['agency_fee'] += ($extension['amount'] - $extension['payment']);

    $row++;
}

$row += 2;

foreach ($totals as $currency => $total) {
    $sheet->setCellValue('A' . $row, 'Totals for ' . $currency);
    $sheet->mergeCells('A' . $row . ':G' . $row);
    $sheet->setCellValue('H' . $row, $currency . ' ' . number_format($total['amount'], 2));
    $sheet->setCellValue('I' . $row, $currency . ' ' . number_format($total['payment'], 2));
    $sheet->setCellValue('J' . $row, $currency . ' ' . number_format($total['agency_fee'], 2));
    $sheet->getStyle('A' . $row . ':K' . $row)->getFont()->setBold(true);
    $row++;
}

foreach(range('A','K') as $col) {
    $sheet->getColumnDimension($col)->setAutoSize(true);
}

$sheet->getStyle('A4:K'.($row-1))->getBorders()->getAllBorders()->setBorderStyle(Border::BORDER_THIN);

$writer = new Xlsx($spreadsheet);

header('Content-Type: application/vnd.openxmlformats-officedocument.spreadsheetml.sheet');
header('Content-Disposition: attachment;filename="extensions_report_' . date('Y-m-d') . '.xlsx"');
header('Cache-Control: max-age=0');

$writer->save('php://output');
exit;
?>