<?php
include 'layouts/session.php';
include 'layouts/dbconn.php';

// Get filter parameters
$agent_id = isset($_GET['agent_id']) ? $_GET['agent_id'] : '';
$service_type = isset($_GET['service_type']) ? $_GET['service_type'] : 'all';
$start_date = isset($_GET['start_date']) ? $_GET['start_date'] : date('Y-m-01');
$end_date = isset($_GET['end_date']) ? $_GET['end_date'] : date('Y-m-t');

// Get selected agent's name if applicable
$selected_agent_name = '';
if (!empty($agent_id)) {
    $agent_name_sql = "SELECT full_name FROM users WHERE user_id = ?";
    $stmt = $conn->prepare($agent_name_sql);
    $stmt->bind_param("i", $agent_id);
    $stmt->execute();
    $result = $stmt->get_result();
    if ($row = $result->fetch_assoc()) {
        $selected_agent_name = $row['full_name'];
    }
}

// Set headers for Excel download
header('Content-Type: application/vnd.ms-excel');
header('Content-Disposition: attachment;filename="agent_services_report_' . date('Y-m-d') . '.xls"');
header('Cache-Control: max-age=0');

echo '<!DOCTYPE html>';
echo '<html>';
echo '<head>';
echo '<meta charset="utf-8">';
echo '<style>';
echo 'table { border-collapse: collapse; width: 100%; }';
echo 'th, td { border: 1px solid black; padding: 8px; text-align: left; }';
echo 'th { background-color: #f2f2f2; }';
echo '.header { font-size: 18px; font-weight: bold; margin-bottom: 10px; }';
echo '.section-header { font-size: 16px; font-weight: bold; margin-top: 20px; margin-bottom: 10px; background-color: #e8e8e8; padding: 5px; }';
echo '.total-row { font-weight: bold; background-color: #f9f9f9; }';
echo '</style>';
echo '</head>';
echo '<body>';

// Header
echo '<div class="header">Agent Services Report</div>';
echo '<div>';
if (!empty($agent_id)) {
    echo 'Agent: ' . htmlspecialchars($selected_agent_name) . ' | ';
}
echo 'Period: ' . date('d M Y', strtotime($start_date)) . ' to ' . date('d M Y', strtotime($end_date));
echo '</div><br>';

// Function to get ticket details
function exportTicketDetails($conn, $agent_id, $start_date, $end_date) {
    $sql = "SELECT td.*, u.full_name as agent_name
            FROM ticket_details td
            JOIN users u ON td.created_by = u.user_id
            WHERE DATE(td.flight_date) BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($agent_id)) {
        $sql .= " AND td.created_by = ?";
        $params[] = $agent_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY td.flight_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Function to get visa applications
function exportVisaApplications($conn, $agent_id, $start_date, $end_date) {
    $sql = "SELECT va.*, u.full_name as agent_name
            FROM visa_applications va
            JOIN users u ON va.created_by = u.user_id
            WHERE va.created_at BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($agent_id)) {
        $sql .= " AND va.created_by = ?";
        $params[] = $agent_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY va.created_at DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Function to get passport extensions
function exportPassportExtensions($conn, $agent_id, $start_date, $end_date) {
    $sql = "SELECT pe.*, u.full_name as agent_name
            FROM passport_extensions pe
            JOIN users u ON pe.created_by = u.user_id
            WHERE pe.extension_date BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($agent_id)) {
        $sql .= " AND pe.created_by = ?";
        $params[] = $agent_id;
        $types .= "i";
    }
    
    $sql .= " ORDER BY pe.extension_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Function to get go home services
function exportGoHomeServices($conn, $agent_name, $start_date, $end_date) {
    $sql = "SELECT ghs.*
            FROM go_home_service ghs
            WHERE ghs.service_date BETWEEN ? AND ?";
    
    $params = [$start_date, $end_date];
    $types = "ss";
    
    if (!empty($agent_name)) {
        $sql .= " AND ghs.agent_name = ?";
        $params[] = $agent_name;
        $types .= "s";
    }
    
    $sql .= " ORDER BY ghs.service_date DESC";
    
    $stmt = $conn->prepare($sql);
    $stmt->bind_param($types, ...$params);
    $stmt->execute();
    return $stmt->get_result();
}

// Initialize totals
$grand_total_revenue = 0;
$grand_total_commission = 0;

// Export Ticket Sales
if ($service_type == 'all' || $service_type == 'tickets') {
    echo '<div class="section-header">Ticket Sales</div>';
    echo '<table>';
    echo '<thead>';
    echo '<tr>';
    echo '<th>Date</th>';
    echo '<th>Agent</th>';
    echo '<th>Customer</th>';
    echo '<th>Ticket Reference</th>';
    echo '<th>Airline</th>';
    echo '<th>Route</th>';
    echo '<th>Ticket Cost</th>';
    echo '<th>Commission</th>';
    echo '<th>Total</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    $tickets = exportTicketDetails($conn, $agent_id, $start_date, $end_date);
    $total_ticket_cost = 0;
    $total_ticket_commission = 0;
    
    if ($tickets && $tickets->num_rows > 0) {
        while ($ticket = $tickets->fetch_assoc()) {
            $total_ticket_cost += $ticket['ticket_cost'];
            $total_ticket_commission += $ticket['agency_fee'];
            
            echo '<tr>';
            echo '<td>' . date('d M Y', strtotime($ticket['flight_date'])) . '</td>';
            echo '<td>' . htmlspecialchars($ticket['agent_name']) . '</td>';
            echo '<td>' . htmlspecialchars($ticket['customer_name']) . '</td>';
            echo '<td>' . htmlspecialchars($ticket['ticket_ref']) . '</td>';
            echo '<td>' . htmlspecialchars($ticket['airline']) . '</td>';
            echo '<td>' . htmlspecialchars($ticket['flight_route']) . '</td>';
            echo '<td>$' . number_format($ticket['ticket_cost'], 2) . '</td>';
            echo '<td>$' . number_format($ticket['agency_fee'], 2) . '</td>';
            echo '<td>$' . number_format($ticket['total_amount'], 2) . '</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="9">No ticket records found</td></tr>';
    }
    
    echo '<tr class="total-row">';
    echo '<td colspan="6">Total</td>';
    echo '<td>$' . number_format($total_ticket_cost, 2) . '</td>';
    echo '<td>$' . number_format($total_ticket_commission, 2) . '</td>';
    echo '<td>$' . number_format($total_ticket_cost + $total_ticket_commission, 2) . '</td>';
    echo '</tr>';
    echo '</tbody>';
    echo '</table><br><br>';
    
    $grand_total_revenue += $total_ticket_cost;
    $grand_total_commission += $total_ticket_commission;
}

// Export Visa Applications
if ($service_type == 'all' || $service_type == 'visas') {
    echo '<div class="section-header">Visa Applications</div>';
    echo '<table>';
    echo '<thead>';
    echo '<tr>';
    echo '<th>Date</th>';
    echo '<th>Agent</th>';
    echo '<th>Client Name</th>';
    echo '<th>Reference Number</th>';
    echo '<th>Amount</th>';
    echo '<th>Payment</th>';
    echo '<th>Commission</th>';
    echo '<th>Status</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    $visas = exportVisaApplications($conn, $agent_id, $start_date, $end_date);
    $total_visa_amount = 0;
    $total_visa_commission = 0;
    
    if ($visas && $visas->num_rows > 0) {
        while ($visa = $visas->fetch_assoc()) {
            $total_visa_amount += $visa['amount'];
            $total_visa_commission += $visa['commission'];
            
            echo '<tr>';
            echo '<td>' . date('d M Y', strtotime($visa['created_at'])) . '</td>';
            echo '<td>' . htmlspecialchars($visa['agent_name']) . '</td>';
            echo '<td>' . htmlspecialchars($visa['client_name']) . '</td>';
            echo '<td>' . htmlspecialchars($visa['reference_number']) . '</td>';
            echo '<td>$' . number_format($visa['amount'], 2) . '</td>';
            echo '<td>$' . number_format($visa['payment'], 2) . '</td>';
            echo '<td>$' . number_format($visa['commission'], 2) . '</td>';
            echo '<td>' . ucfirst($visa['status']) . '</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="8">No visa records found</td></tr>';
    }
    
    echo '<tr class="total-row">';
    echo '<td colspan="4">Total</td>';
    echo '<td>$' . number_format($total_visa_amount, 2) . '</td>';
    echo '<td></td>';
    echo '<td>$' . number_format($total_visa_commission, 2) . '</td>';
    echo '<td></td>';
    echo '</tr>';
    echo '</tbody>';
    echo '</table><br><br>';
    
    $grand_total_revenue += $total_visa_amount;
    $grand_total_commission += $total_visa_commission;
}

// Export Passport Extensions
if ($service_type == 'all' || $service_type == 'extensions') {
    echo '<div class="section-header">Passport Extensions</div>';
    echo '<table>';
    echo '<thead>';
    echo '<tr>';
    echo '<th>Date</th>';
    echo '<th>Agent</th>';
    echo '<th>Client Name</th>';
    echo '<th>Passport Number</th>';
    echo '<th>Type</th>';
    echo '<th>Amount Received</th>';
    echo '<th>Service Payment</th>';
    echo '<th>Agency Fee</th>';
    echo '<th>Status</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    $extensions = exportPassportExtensions($conn, $agent_id, $start_date, $end_date);
    $total_ext_amount = 0;
    $total_ext_agency_fee = 0;
    
    if ($extensions && $extensions->num_rows > 0) {
        while ($ext = $extensions->fetch_assoc()) {
            $total_ext_amount += $ext['amount_received'];
            $total_ext_agency_fee += $ext['agency_fee'];
            
            echo '<tr>';
            echo '<td>' . date('d M Y', strtotime($ext['extension_date'])) . '</td>';
            echo '<td>' . htmlspecialchars($ext['agent_name']) . '</td>';
            echo '<td>' . htmlspecialchars($ext['client_name']) . '</td>';
            echo '<td>' . htmlspecialchars($ext['passport_number']) . '</td>';
            echo '<td>' . ucfirst($ext['service_type']) . '</td>';
            echo '<td>$' . number_format($ext['amount_received'], 2) . '</td>';
            echo '<td>$' . number_format($ext['service_payment'], 2) . '</td>';
            echo '<td>$' . number_format($ext['agency_fee'], 2) . '</td>';
            echo '<td>' . ucfirst($ext['payment_status']) . '</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="9">No extension records found</td></tr>';
    }
    
    echo '<tr class="total-row">';
    echo '<td colspan="5">Total</td>';
    echo '<td>$' . number_format($total_ext_amount, 2) . '</td>';
    echo '<td></td>';
    echo '<td>$' . number_format($total_ext_agency_fee, 2) . '</td>';
    echo '<td></td>';
    echo '</tr>';
    echo '</tbody>';
    echo '</table><br><br>';
    
    $grand_total_revenue += $total_ext_amount;
    $grand_total_commission += $total_ext_agency_fee;
}

// Export Go Home Services
if ($service_type == 'all' || $service_type == 'go_home') {
    echo '<div class="section-header">Go Home Services</div>';
    echo '<table>';
    echo '<thead>';
    echo '<tr>';
    echo '<th>Date</th>';
    echo '<th>Agent</th>';
    echo '<th>Client Name</th>';
    echo '<th>Mother Name</th>';
    echo '<th>DOB</th>';
    echo '<th>Reference</th>';
    echo '<th>Price</th>';
    echo '<th>Office Fee</th>';
    echo '<th>Status</th>';
    echo '</tr>';
    echo '</thead>';
    echo '<tbody>';
    
    $goHomeServices = exportGoHomeServices($conn, $selected_agent_name, $start_date, $end_date);
    $total_go_home_price = 0;
    $total_go_home_fee = 0;
    
    if ($goHomeServices && $goHomeServices->num_rows > 0) {
        while ($ghs = $goHomeServices->fetch_assoc()) {
            $total_go_home_price += $ghs['price'];
            $total_go_home_fee += $ghs['office_fee'];
            
            echo '<tr>';
            echo '<td>' . date('d M Y', strtotime($ghs['service_date'])) . '</td>';
            echo '<td>' . htmlspecialchars($ghs['agent_name']) . '</td>';
            echo '<td>' . htmlspecialchars($ghs['client_name']) . '</td>';
            echo '<td>' . htmlspecialchars($ghs['mother_name']) . '</td>';
            echo '<td>' . date('d M Y', strtotime($ghs['date_of_birth'])) . '</td>';
            echo '<td>' . htmlspecialchars($ghs['reference_number']) . '</td>';
            echo '<td>$' . number_format($ghs['price'], 2) . '</td>';
            echo '<td>$' . number_format($ghs['office_fee'], 2) . '</td>';
            echo '<td>' . ucfirst($ghs['status']) . '</td>';
            echo '</tr>';
        }
    } else {
        echo '<tr><td colspan="9">No go home service records found</td></tr>';
    }
    
    echo '<tr class="total-row">';
    echo '<td colspan="6">Total</td>';
    echo '<td>$' . number_format($total_go_home_price, 2) . '</td>';
    echo '<td>$' . number_format($total_go_home_fee, 2) . '</td>';
    echo '<td></td>';
    echo '</tr>';
    echo '</tbody>';
    echo '</table><br><br>';
    
    $grand_total_revenue += $total_go_home_price;
    $grand_total_commission += $total_go_home_fee;
}

// Overall Summary
if ($service_type == 'all') {
    echo '<div class="section-header">Overall Summary</div>';
    echo '<table>';
    echo '<tr class="total-row">';
    echo '<td>Total Revenue</td>';
    echo '<td>$' . number_format($grand_total_revenue, 2) . '</td>';
    echo '</tr>';
    echo '<tr class="total-row">';
    echo '<td>Total Commission</td>';
    echo '<td>$' . number_format($grand_total_commission, 2) . '</td>';
    echo '</tr>';
    echo '</table>';
}

echo '</body>';
echo '</html>';
?>