<?php
session_start();
include 'layouts/dbconn.php';

if ($_SERVER['REQUEST_METHOD'] != 'POST') {
    http_response_code(405);
    echo json_encode(['success' => false, 'message' => 'Method not allowed']);
    exit;
}

try {
    // Validate required fields
    $required_fields = [
        'extension_date', 'client_name', 'passport_number', 
        'country_of_issue', 'service_type', 'validity_period',
        'expiry_date', 'amount_received', 'service_payment', 
        'payment_status', 'currency'
    ];

    foreach ($required_fields as $field) {
        if (empty($_POST[$field])) {
            throw new Exception("$field is required");
        }
    }

    // Validate dates
    $extension_date = new DateTime($_POST['extension_date']);
    $expiry_date = new DateTime($_POST['expiry_date']);

    if ($expiry_date <= $extension_date) {
        throw new Exception("Expiry date must be after extension date");
    }

    // Validate amounts
    $amount_received = floatval($_POST['amount_received']);
    $service_payment = floatval($_POST['service_payment']);

    if ($amount_received < 0 || $service_payment < 0) {
        throw new Exception("Amounts cannot be negative");
    }

    if ($service_payment > $amount_received) {
        throw new Exception("Service payment cannot be greater than amount received");
    }

    // Prepare the SQL statement
    $sql = "INSERT INTO passport_extensions (
                extension_date, client_name, passport_number, 
                country_of_issue, service_type, validity_period,
                expiry_date, amount_received, service_payment,
                payment_status, notes, created_by, currency
            ) VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?)";

    $stmt = $conn->prepare($sql);
    
    if (!$stmt) {
        throw new Exception("Failed to prepare statement: " . $conn->error);
    }

    // Bind parameters
    $stmt->bind_param(
        "sssssisddssis",
        $_POST['extension_date'],
        $_POST['client_name'],
        $_POST['passport_number'],
        $_POST['country_of_issue'],
        $_POST['service_type'],
        $_POST['validity_period'],
        $_POST['expiry_date'],
        $amount_received,
        $service_payment,
        $_POST['payment_status'],
        $_POST['notes'],
        $_SESSION['user_id'],
        $_POST['currency']
    );

    // Execute the statement
    if (!$stmt->execute()) {
        throw new Exception("Failed to save extension: " . $stmt->error);
    }

    // Log the action
    $log_sql = "INSERT INTO audit_logs (user_id, action, log_details, ip_address, user_agent) 
                VALUES (?, 'CREATE_EXTENSION', ?, ?, ?)";
    $log_stmt = $conn->prepare($log_sql);
    $log_details = "Created passport extension for " . $_POST['client_name'];
    $log_stmt->bind_param(
        "isss", 
        $_SESSION['user_id'],
        $log_details,
        $_SERVER['REMOTE_ADDR'],
        $_SERVER['HTTP_USER_AGENT']
    );
    $log_stmt->execute();

    echo json_encode([
        'success' => true,
        'message' => 'Passport extension saved successfully',
        'extension_id' => $conn->insert_id
    ]);

} catch (Exception $e) {
    http_response_code(400);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}

$conn->close();
?>